<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;
use App\Employee;
use App\Department;
use App\Designation;
use App\Policy;
use App\Payroll;

class EmployeeController extends Controller {

    protected $employee;

    public function __construct() {
        $this->employee = new Employee();
    }

    //Get All the Employee
    public function allEmployees() {

        $allEmployees = Employee::with('department','designation')->orderBy('created_at', 'desc')->get();

        return view('payroll/employee.index', compact('allEmployees'));
    }

    //show the add new Employee form
    public function addEmployeeForm() {
        
        $departments = Department::where('status',true)->orderBy('name','asc')->get();
        $designations = Designation::where('status',true)->orderBy('name','asc')->get();
        $policies = Policy::orderBy('name','asc')->get();

        return view('payroll/employee.add-employee',compact('departments','designations','policies'));
    }

    //insert new Employee record
    public function insertEmployee(Request $request) {

        $validator = Validator::make($request->all(), [
                    'employee_id'=>'required',
                    'name' => 'required|regex:/^[a-zA-Z\s]+$/u',
                    'basic_salary' => 'required|numeric|min:0',
                    'designation_id' => 'required',
                    'department_id' => 'required',
                    'status' => 'required',
                    'employee_type' => 'required',
                    'working_hours' => 'numeric|min:0|nullable',
                    'working_days' => 'numeric|min:0|nullable',
                    'bank_name' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'account_name' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'email' => 'nullable|email',
                    'phone_numnber' => 'nullable',
                    'emer_person' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'emer_phone_number'=> 'nullable',
                    'emer_email' => 'nullable|email',
        ]);

        if ($validator->fails()) {
            return redirect('/employees/add')
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //create new Employee
        $employee = $this->employee->insertNewEmployee($request);

        return redirect()
                        ->route('employee.detail', ['id' => $employee->id])
                        ->with([
                            'message' => 'Employee created successfully!',
                            'alert-type' => 'info'
        ]);
    }

    //show the detail of the Employee
    public function detailEmployee($id) {

        $employee = Employee::with('department', 'designation','policies','payroll')->where('id', $id)->first();
        

        return view('payroll/employee.detail-employee', compact('employee'));
    }

    //show the edit Employee form
    public function editEmployeeForm($id) {

        $employee = Employee::with('department', 'designation','policies')->where('id', $id)->first();
       
        $departments = Department::where('status',true)->orderBy('name','asc')->get();
        $designations = Designation::where('status',true)->orderBy('name','asc')->get();
        $policies = Policy::orderBy('name','asc')->get();

        return view('payroll/employee.edit-employee', compact('employee','departments','designations','policies'));
    }

    //update the Employee record
    public function updateEmployee(Request $request) {

        $validator = Validator::make($request->all(), [
                    'employee_id'=>'required',
                    'name' => 'required|regex:/^[a-zA-Z\s]+$/u',
                    'basic_salary' => 'required|numeric|min:0',
                    'designation_id' => 'required',
                    'department_id' => 'required',
                    'status' => 'required',
                    'employee_type' => 'required',
                    'working_hours' => 'numeric|min:0|nullable',
                    'working_days' => 'numeric|min:0|nullable',
                    'bank_name' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'account_name' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'email' => 'nullable|email',
                    'phone_numnber' => 'nullable',
                    'emer_person' => 'nullable|regex:/^[a-zA-Z\s]+$/u',
                    'emer_phone_number'=> 'nullable',
                    'emer_email' => 'nullable|email',
        ]);

        if ($validator->fails()) {
            return redirect('/employees/edit/' . $request->id)
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //update Employee
        $this->employee->updateEmployee($request);

        return redirect()
                       ->route('employee.detail', ['id' => $request->id])
                        ->with([
                            'message' => 'Employee updated successfully!',
                            'alert-type' => 'info'
        ]);
    }

    //delete the specific Employee
    public function deleteEmployee($id) {

        Employee::where('id', $id)->delete();

        return redirect()->route('employee.all')->with([
                    'message' => 'Employee deleted Successfully!',
                    'alert-type' => 'info',
        ]);
    }
    
    //get the payroll record of all the active employees
    public function getPayrollSheet(){
        
        $employees = Employee::where('status', Employee::active)->with('policies')->orderBy('name','asc')
                ->get();
        
        //mark those whose salaries is paid
        $salaryPaid = array();
        foreach($employees as $employee) {
            $salaryPaid[$employee->id] = $this->employee->checkSalaryPaidOrNot($employee->id);
        }
        
        return view('payroll.payroll',compact('employees','salaryPaid'));
    }
    
    //get the payroll history of all the employees
    public function getPayrollHistory(){
        
        $payrolls = Payroll::with('employee')->orderBy('created_at','desc')->get();
        
        $totalPay = Payroll::sum('net_pay');
        
        return view('payroll.payroll-history',compact('payrolls','totalPay'));
    }
    
    //run the payroll of the employee 
    public function runPayrollByEmployee($id) {

        $result = $this->employee->runPayrollByEmployeeId($id);

        if ($result == true) {
            return redirect()
                            ->route('employee.payroll.sheet')
                            ->with([
                                'message' => 'Payroll run successfully!',
                                'alert-type' => 'info'
            ]);
        } else {
            return redirect()
                            ->route('employee.payroll.sheet')
                            ->with([
                                'message' => 'Emplyee salary already paid for this month',
                                'alert-type' => 'warning'
            ]);
        }
    }

}
