<?php

namespace App\Http\Controllers;


use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Validator;
use App\Expense;
use App\ExpenseCategory;
use App\Vendor;
use App\ExpensePayment;
use App\CompanyBank;
use App\ChartOfAccount;

class ExpenseController extends Controller {

    protected $expense;

    public function __construct() {
        $this->expense = new Expense();
    }

    //Get All the expenses
    public function allExpenses() {

        $data['allExpenses'] = Expense::orderBy('created_at', 'desc')->get();

        $data['amount'] = Expense::orderBy('created_at', 'desc')->sum('amount');

        $data['vendors'] = Vendor::where('status',true)->get();

        $data['categories'] = ChartOfAccount::orderBy('number','ASC')->get();

        return view('expense/bill.index', $data);
    }

    public function updateTable(Request $request) {

        $filters = array();

        if($request->vendor != 0)
            $filters['vendor'] = $request->vendor;
        if($request->category != 0)
            $filters['category'] = $request->category;
        if($request->status != 0)
            $filters['status'] = $request->status;
        if($request->date != 0)
            $filters['date'] = $request->date;

        $data = $this->expense->updateTable($filters);

        return view('expense/bill.update-table',$data);
    }

    //show the add new expense  form
    public function addExpenseForm() {

        $data['vendors'] = Vendor::where('status',true)->get();

        $data['accounts'] = ChartOfAccount::orderBy('number','ASC')->get();

        return view('expense/bill.add-bill',$data);
    }

    //insert new expense record
    public function insertExpense(Request $request) {

        $validator = Validator::make($request->all(), [
                    'category_id' => 'required',
                    'vendor_id' => 'required',
                    'billed_at'=>'required',
                    'due_at'=>'required',
                    'amount'=>'required|numeric|min:1'
        ]);

        if ($validator->fails()) {
            return redirect('/expense/add')
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //create new Bill
        //$expense = Expense::create(Input::all());
        $input=$request->all();
        $expense = Expense::create($input);

        if ($request->hasfile('image')) {
            $file = $request->file('image');
;
            $name = $expense->id . '-' . $file->getClientOriginalName();
            $file->move(public_path() . '/uploads/', $name);

            Expense::where('id',$expense->id)->update(['image_url' => $name]);
        }


        

        if($request->receipt) {

        $payment = ExpensePayment::create(['bill_id'=>$expense->id,'amount'=>$request->pay_amount,'date'=>$expense->billed_at,'account_id'=>$request->account_id,'payment_method'=>$request->payment_method,'notes'=>$request->payment_description]);

        //check cash or bank transaction
        if($request->payment_mode != ExpensePayment::CASH)
            ExpensePayment::where('id',$payment->id)->update(['is_reconciled'=>false]);

        $bill = Expense::where('id',$expense->id)->first();

        $paid_amount = ExpensePayment::where('bill_id',$expense->id)->sum('amount');

        if($paid_amount  == $bill->amount){
            Expense::where('id',$expense->id)->update(['status'=>\App\Expense::PAID]);}
        else {
            Expense::where('id',$expense->id)->update(['status'=>\App\Expense::PARTIALLY]);}


        
        }
        return redirect()
                        ->route('expense.detail',['id'=>$expense->id])
                        ->with([
                            'message' => 'Bill created successfully!',
                            'alert-type' => 'info'
        ]);
    }

    //show the detail of the Expense
    public function detailExpense($id) {

        $expense = Expense::where('id', $id)->first();

        return view('expense/bill.detail-bill', compact('expense'));
    }

    //show the edit expense form
    public function editExpenseForm($id) {

        $data['expense'] = Expense::where('id', $id)->first();

        $data['vendors'] = Vendor::where('status',true)->get();

        $data['categories'] = ExpenseCategory::where('status',true)->get();

        return view('expense/bill.edit-bill',$data);
    }

    //update the Expense record
    public function updateExpense(Request $request) {

        $validator = Validator::make($request->all(), [
                    'category_id' => 'required',
                    'vendor_id' => 'required',
                    'billed_at'=>'required',
                    'due_at'=>'required',
                    'amount'=>'required|numeric|min:1'

        ]);

        if ($validator->fails()) {
            return redirect('/expense/edit/' . $request->id)
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //update Expense
        Expense::where('id',$request->id)->update(Input::except('_token'));

        if ($request->hasfile('image')) {
            $file = $request->file('image');

            $name = $request->id . '-' . $file->getClientOriginalName();
            $file->move(public_path() . '/uploads/', $name);

            Expense::where('id',$request->id)->update(['image_url' => $name]);


        }



        return redirect()
                        ->route('expense.detail',['id'=>$request->id])
                        ->with([
                            'message' => 'Bill updated successfully!',
                            'alert-type' => 'info'
        ]);
    }

    //delete the specific expense
    public function deleteExpense($id) {

        $exepense = Expense::find($id);


        Expense::where('id', $id)->delete();

        return redirect()->route('expense.all')->with([
                    'message' => 'Bill deleted Successfully!',
                    'alert-type' => 'info',
        ]);
    }


    //Get All the payments
    public function allPayments() {

        $data['allPayments'] = ExpensePayment::orderBy('created_at', 'desc')->get();
        $data['totalAmount'] = ExpensePayment::orderBy('created_at', 'desc')->sum('amount');

        $data['vendors'] = Vendor::where('status',true)->get();
        $data['accounts'] = ChartOfAccount::orderBy('number','ASC')->get();
        $data['categories'] = ChartOfAccount::orderBy('number','ASC')->get();

        return view('expense/payment.index', $data);
    }

    public function updatePaymentTable(Request $request) {

        $filters = array();

        if($request->vendor != 0)
            $filters['vendor'] = $request->vendor;
        if($request->category != 0)
            $filters['category'] = $request->category;
        if($request->account != 0)
            $filters['account'] = $request->account;
        if($request->date != 0)
            $filters['date'] = $request->date;

        $data = $this->expense->updatePaymentTable($filters);

        return view('expense/payment.update-table',$data);
    }

    //show the add new payment  form
    public function addPaymentForm() {

        $data['vendors'] = Vendor::where('status',true)->get();
        $data['accounts'] = ChartOfAccount::orderBy('number','ASC')->get();


        return view('expense/payment.add-payment',$data);
    }

    public function getVendorOpenBill(Request $request) {

        $data['bills'] = Expense::where([['vendor_id',$request->vendor_id],['status','!=',\App\Expense::PAID]])->get();

        $remAmount = array();

        foreach($data['bills'] as $bill) {
            $remAmount[$bill->id] = $bill->amount - ExpensePayment::where('bill_id',$bill->id)->sum('amount');
        }

        $data['remAmount'] = $remAmount;

        return view('expense/payment.update-bill-table',$data);
    }

    //insert new payment record
    public function insertPayment(Request $request) {

        $validator = Validator::make($request->all(), [
                    'bill_id' => 'required',
                    'vendor_id' => 'required',
                    'account_id' => 'required',
                    'amount'=>'required',
                    'date'=>'required',
                    'account_id'=>'required'
        ]);

        if ($validator->fails()) {
            return redirect('/payment/add')
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //create new Bill
        $payment = ExpensePayment::create(Input::except('vendor_id','due'));

        //check cash or bank transaction
        if($request->payment_mode != ExpensePayment::CASH)
            ExpensePayment::where('id',$payment->id)->update(['is_reconciled'=>false]);

        $bill = Expense::where('id',$request->bill_id)->first();

        $paid_amount = ExpensePayment::where('bill_id',$request->bill_id)->sum('amount');

        if($paid_amount  == $bill->amount){
            Expense::where('id',$request->bill_id)->update(['status'=>\App\Expense::PAID]);}
        else {
            Expense::where('id',$request->bill_id)->update(['status'=>\App\Expense::PARTIALLY]);}


        

        return redirect()->route('payment.all')->with([
                    'message' => 'Payment added Successfully!',
                    'alert-type' => 'info',
        ]);
    }

    //show the edit expense form
    public function editPaymentForm($id) {

        $data['payment'] = ExpensePayment::where('id', $id)->first();

        $data['vendors'] = Vendor::where('status',true)->get();
        $data['accounts'] = ChartOfAccount::orderBy('number','ASC')->get();

        return view('expense/payment.edit-payment',$data);
    }

    //update the Expense record
    public function updatePayment(Request $request) {

        $validator = Validator::make($request->all(), [
                    // 'category_id' => 'required',
                    // 'vendor_id' => 'required',
                    'account_id' => 'required',
                    'amount'=>'required',
                    'date'=>'required',

        ]);

        if ($validator->fails()) {
            return redirect('/payment/edit/' . $request->id)
                            ->withErrors($validator)
                            ->withInput()
                            ->with([
                                'message' => 'Invalid Inputs!',
                                'alert-type' => 'error',
            ]);
        }

        //update Expense
        ExpensePayment::where('id',$request->id)->update(Input::except('_token'));

        if ($request->hasfile('file')) {
            $file = $request->file('file');
            

            $name = 'payment-' . $request->id . '.' . $file->getClientOriginalExtension();
            $file->move(public_path() . '/uploads/', $name);

            ExpensePayment::where('id',$request->id)->update(['file' => $name]);
        }

        
        

        return redirect()->route('payment.all')->with([
                    'message' => 'Payment updated Successfully!',
                    'alert-type' => 'info',
        ]);
    }

    //delete the specific expense
    public function deletePayment($id) {

        $expense = ExpensePayment::where('id',$id)->first();

        
        ExpensePayment::where('id', $id)->delete();

        Expense::where('id',$expense->bill_id)->update(['status'=>\App\Expense::UNPAID]);

        return redirect()->route('payment.all')->with([
                    'message' => 'Payment deleted Successfully!',
                    'alert-type' => 'info',
        ]);
    }

    public function printVoucher($id) {

        $data['payment'] = ExpensePayment::where('id',$id)->first();

         $digit = new \NumberFormatter("en", \NumberFormatter::SPELLOUT);
         $data['words'] = $digit->format($data['payment']->amount);

         if($data['payment']->bill->expense_type == Expense::CONSTRUCTION)
            return view('expense/payment.construction-voucher',$data);
        else
            return view('expense/payment.service-voucher',$data);

        
    }

    //show the detail of the Payment
    public function detailPayment($id) {

        $data['payment'] = ExpensePayment::where('id',$id)->first();

        return view('expense/payment.detail-bill', compact('payment'));
    }

}
